/* vi: set sw=4 ts=4: */
/*
 * Mini rtklogd implementation for busybox (based on klogd implementation)
 *
 * Copyright (C) 2001 by Gennady Feldman <gfeldman@gena01.com>.
 * Changes: Made this a standalone busybox module which uses standalone
 * syslog() client interface.
 *
 * Copyright (C) 1999-2004 by Erik Andersen <andersen@codepoet.org>
 *
 * Copyright (C) 2000 by Karl M. Hegbloom <karlheg@debian.org>
 *
 * "circular buffer" Copyright (C) 2000 by Gennady Feldman <gfeldman@gena01.com>
 *
 * Portions Copyright (c) 2020-2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under GPLv2 or later, see file LICENSE in this source tree.
 */

//config:config RTKLOGD
//config:	bool "rtklogd"
//config:	default n
//config:	help
//config:	rtklogd is equivalent to klogd for the RTK


//applet:IF_RTKLOGD(APPLET(rtklogd, BB_DIR_SBIN, BB_SUID_DROP))

//kbuild:lib-$(CONFIG_RTKLOGD) += rtklogd.o

//usage:#define rtklogd_trivial_usage
//usage:       "[-n]"
//usage:#define rtklogd_full_usage "\n\n"
//usage:       "RTK logger\n"
//usage:     "\n	-n	Run in foreground"

#include "libbb.h"
#include <syslog.h>

// SAV526 devices
#define _PATH_RTKLOG "/proc/rtos/log"
// SAV533/537 devices
#define _PATH_ALT_RTKLOG "/proc/dualos/log"

static const char *rtk_log_p = _PATH_RTKLOG;

static void rtklogd_open(void)
{
    /* Can't do a one-time open.  The fd becomes invalid */
}

static int rtklogd_read(char *bufp, int len)
{
	int rtklogfd = open(rtk_log_p, O_RDONLY);
	int n;
	if (rtklogfd < 0) {
		if (rtk_log_p == (const char *)_PATH_RTKLOG) {
			rtk_log_p = _PATH_ALT_RTKLOG;
		} else {
			rtk_log_p = _PATH_RTKLOG;
		}
		rtklogfd = xopen(rtk_log_p, O_RDONLY);
	}
	n = rtklogfd;
	if (rtklogfd >= 0) {
		n = read(rtklogfd, bufp, len);
		close(rtklogfd);
	}
	return n;
}
#define READ_ERROR "rtklog read error"

static void rtklogd_close(void)
{
}

char rtk_log_buffer[BUFSIZ];
enum {
	RTKLOGD_LOGBUF_SIZE = sizeof(rtk_log_buffer),
	OPT_FOREGROUND = (1 << 0),
};

int rtklogd_main(int argc, char **argv) MAIN_EXTERNALLY_VISIBLE;
int rtklogd_main(int argc UNUSED_PARAM, char **argv)
{
	int opt;
	int used;

	opt = getopt32(argv, "n");

	if (!(opt & OPT_FOREGROUND)) {
		bb_daemonize_or_rexec(DAEMON_CHDIR_ROOT, argv);
	}

	logmode = LOGMODE_SYSLOG;

	/* rtklogd_open() before openlog(), since it might use fixed fd ,
	 * and openlog() also may use the same fd 4 if we swap them:
	 */
	rtklogd_open();
	openlog("rtklog", 0, LOG_INFO);

	signal(SIGHUP, SIG_IGN);
	/* We want rtklogd_read to not be restarted, thus _norestart: */
	bb_signals_norestart(BB_FATAL_SIGS, record_signo);

	syslog(LOG_NOTICE, "rtklogd started: %s", bb_banner);

	write_pidfile(CONFIG_PID_FILE_PATH "/rtklogd.pid");

	used = 0;
	while (!bb_got_signal) {
		int n;
		char *start;

		/* "2 -- Read from the log." */
		start = rtk_log_buffer + used;
		n = rtklogd_read(start, RTKLOGD_LOGBUF_SIZE-1 - used);
		if (n < 0) {
			if (errno == EINTR)
				continue;
			bb_perror_msg(READ_ERROR);
			break;
		}
		if (n == 0 || ((n == 1) && (*start == '\n'))) {
			/* Prevent CPU load at 100% - file read doesn't block and may return LF for no data */
			sleep(1);
			continue;
		}
		start[n] = '\0';

		/* Process each newline-terminated line in the buffer */
		start = rtk_log_buffer;
		while (1) {
			char *newline = strchrnul(start, '\n');

			if (*newline == '\0') {
				/* This line is incomplete */

				/* move it to the front of the buffer */
				overlapping_strcpy(rtk_log_buffer, start);
				used = newline - start;
				if (used < RTKLOGD_LOGBUF_SIZE-1) {
					/* buffer isn't full */
					break;
				}
				/* buffer is full, log it anyway */
				used = 0;
				newline = NULL;
			} else {
				*newline++ = '\0';
			}

			/* Log (only non-empty lines) */
			if (*start)
				syslog(LOG_INFO, "%s", start);

			if (!newline)
				break;
			start = newline;
		}
	}

	rtklogd_close();
	syslog(LOG_NOTICE, "rtklogd: exiting");
	remove_pidfile(CONFIG_PID_FILE_PATH "/rtklogd.pid");
	if (bb_got_signal)
		kill_myself_with_sig(bb_got_signal);
	return EXIT_FAILURE;
}
